;***********************************************************************
;*                                                                     *
;*  ARRAY.ASP  (C) 1990 DATASTORM TECHNOLOGIES, INC.                   *
;*                                                                     *
;*  Purpose  : This ASPECT module provides procedures for the          *
;*     manipulation of an integer or character array in 1, 2, and 3    *
;*     dimensions.  The maximum number of integer elements allowed is  *
;*     200; the maximum number of character elements allowed is 400.   *
;*     However, this module can be easily modified to expand this      *
;*     range, or to add functionality for handling more than one       *
;*     array. The procedures are called using the ASPECT command-like  *
;*     interface provided by the macros which are defined below.       *
;*                                                                     *
;*  Commands : The following command descriptions use the convention   *
;*     that X_DIM, Y_DIM, and Z_DIM are all integer type values.       *
;*     There are two forms each of the get and put commands: one for   *
;*     integer type arrays and one for character type arrays.  These   *
;*     forms perform identical functions, but allow you to create more *
;*     "readable" code depending on the type of array you're using.    *
;*                                                                     *
;*                                                                     *
;*     SETARRAY TYPEINT|TYPECHAR X_DIM Y_DIM Z_DIM                     *
;*                                                                     *
;*     This command MUST be called before an array can be used.        *
;*     It sets the type (integer or character) and the dimensions      *
;*     of the array.  All dimensions must have values >= 1.  For a     *
;*     1-dimensional array, set X_DIM to the desired value and Y_DIM   *
;*     and Z_DIM to 1.  For a 2-dimensional array, SET X_DIM and Y_DIM *
;*     to the desired values and Z_DIM to 1.  For a 3-dimensional      *
;*     array set all three dimensions to the desired values. Note that *
;*     the value 1 is not considered as an additional dimension unless *
;*     it is followed by another dimension value > 1.                  *
;*                                                                     *
;*                                                                     *
;*     PUTINT X_DIM integer            GETINT X_DIM &intvar            *
;*     PUTCHAR X_DIM integer           GETCHAR X_DIM &intvar           *
;*                                                                     *
;*     These commands work with a 1-dimensional array to put or get    *
;*     a value at a specified index.  Note that the get form of these  *
;*     commands require an integer variable with an ampersand (&)      *
;*     prepended.  This is necessary because these commands are        *
;*     actually procedure calls and require the ampersand to assign    *
;*     a value to the variable.                                        *
;*                                                                     *
;*                                                                     *
;*     PUTINT2 X_DIM Y_DIM integer     GETINT2 X_DIM Y_DIM &integer    *
;*     PUTCHAR2 X_DIM Y_DIM integer    GETCHAR2 X_DIM Y_DIM &integer   *
;*                                                                     *
;*     These commands work with a 2-dimensional array to put or get    *
;*     a value at a specified index.                                   *
;*                                                                     *
;*                                                                     *
;*     PUTINT3 X_DIM Y_DIM Z_DIM integer                               *
;*     GETINT3 X_DIM Y_DIM Z_DIM &intvar                               *
;*     PUTCHAR3 X_DIM Y_DIM Z_DIM integer                              *
;*     GETCHAR3 X_DIM Y_DIM Z_DIM &intvar                              *
;*                                                                     *
;*     These commands work with a 3-dimensional array to put or get    *
;*     a value at a specified index.                                   *
;*                                                                     *
;***********************************************************************

;***********************************************************************
;* Defining symbol ADEBUG will provied debugging information while     *
;* you are developing your scripts.  It will generated error messages  *
;* at run time if you attempt to define an array with invalid values   *
;* or one that is larger than can be supported, or if you attempt to   *
;* access an array with an invalid number of indexes or indexes that   *
;* are out of range with those that you provided.                      *
;*                                                                     *
;* With ADEBUG not defined, these procedures are very lenient with how *
;* arrays are defined and accessed.  BEWARE that placing values into   *
;* an array index which is out of range can have unpredictable side    *
;* effects.                                                            *
;*                                                                     *
;* When your scripts are running without errors, you can disable the   *
;* the debugging information by commenting out the define statement.   *
;***********************************************************************

;define ADEBUG                         ; enables debugging information


;***********************************************************************
;* Procedure call macros which allow procedures in this module to be   *
;* called as if they were ASPECT commands                              *
;***********************************************************************

define setarray call asetarray with

define putint call arrayio with 0 1 0 0
define getint call arrayio with 1 1 0 0

define putint2 call arrayio with 0 2 0
define getint2 call arrayio with 1 2 0

define putint3 call arrayio with 0 3
define getint3 call arrayio with 1 3

define putchar putint
define getchar getint

define putchar2 putint2
define getchar2 getint2

define putchar3 putint3
define getchar3 getint3


;***********************************************************************
;* Defines and global variables for this module                        *
;***********************************************************************

define TYPECHAR 1                      ; size of character type element
define TYPEINT  2                      ; size of integer type element
define ASTRSIZE 80                     ; max size of a string variable
define NUMASTR  5                      ; number of strings used for array

string array0,array1,array2,array3,array4  ; strings for array contents

integer ELEMSIZE                       ; element size (1 or 2)
integer STRELEM                        ; number of elements per string
integer MAXELEM                        ; max number of elements supported
integer AMAXNDX                        ; max index based on dimensions
integer ADIMEN                         ; number of specified dimensions
integer X_SIZE                         ; size of first dimension
integer Y_SIZE                         ; size of second dimension
integer Z_SIZE                         ; size of third dimension


;***********************************************************************
;* This procedure sets the type and dimensions of the array and must   *
;* be called before an array can be used                               *
;*                                                                     *
;* Each dimension should be >= 1                                       *
;* 1-dimensional array: set x to the desired size and y = z = 1        *
;* 2-dimensional array: set x and y to the desired sizes and z = 1     *
;* 3-dimensional array: set x, y, and z to the desired sizes           *
;***********************************************************************

proc asetarray
intparm atype,x,y,z

   ELEMSIZE = atype
   STRELEM  = ASTRSIZE / ELEMSIZE
   MAXELEM  = NUMASTR * STRELEM
   X_SIZE   = x
   Y_SIZE   = y
   Z_SIZE   = z
   AMAXNDX  = X_SIZE * Y_SIZE * Z_SIZE
   if(Z_SIZE > 1)
      ADIMEN = 3
   elseif(Y_SIZE > 1)
      ADIMEN = 2
   else
      ADIMEN = 1
   endif
   $ifdef ADEBUG
      if((ELEMSIZE <= 0) || (ELEMSIZE > 2))
         message "ERROR: Invalid array type specified"
      endif
      if((AMAXNDX <= 0) || (AMAXNDX > MAXELEM))
         message "ERROR: Array dimensions out of range"
      endif
   $endif
endproc


;***********************************************************************
;* Calculate storage mapping index based on array indexes  and set     *
;* or retrieve value at calculated index                               *
;***********************************************************************

proc arrayio
intparm mode,access,x,y,z,value
   integer aindex,asindex
   string array

   $ifdef ADEBUG
      if(access != ADIMEN)
         message "ERROR: Incorrect number of array indexes specified"
      endif
   $endif
   switch access
      case 1
         aindex = z
         $ifdef ADEBUG
            call achkndx with 1,z,X_SIZE
         $endif
      endcase
      case 2
         aindex = z + Y_SIZE * y
         $ifdef ADEBUG
            call achkndx with 1,y,X_SIZE
            call achkndx with 2,z,Y_SIZE
         $endif
      endcase
      case 3
         aindex = z + Z_SIZE * (y + Y_SIZE * x)
         $ifdef ADEBUG
            call achkndx with 1,x,X_SIZE
            call achkndx with 2,y,Y_SIZE
            call achkndx with 3,z,Z_SIZE
         $endif
      endcase
   endswitch
   asindex = aindex / STRELEM
   aindex = (aindex % STRELEM) * ELEMSIZE
   if(mode)
      switch asindex
         case 0
            substr array array0 aindex ELEMSIZE
         endcase
         case 1
            substr array array1 aindex ELEMSIZE
         endcase
         case 2
            substr array array2 aindex ELEMSIZE
         endcase
         case 3
            substr array array3 aindex ELEMSIZE
         endcase
         case 4
            substr array array4 aindex ELEMSIZE
         endcase
      endswitch
      strpeek array 0 value
      if(ELEMSIZE == 2)
         value <<= 8
         strpeek array 1 access
         value |= access
      endif
   else
      if(ELEMSIZE == 2)
         access = value & 0xFF
         strpoke array 1 access
         value >>= 8
      endif
      strpoke array 0 value
      switch asindex
         case 0
            strupdt array0 array aindex ELEMSIZE
         endcase
         case 1
            strupdt array1 array aindex ELEMSIZE
         endcase
         case 2
            strupdt array2 array aindex ELEMSIZE
         endcase
         case 3
            strupdt array3 array aindex ELEMSIZE
         endcase
         case 4
            strupdt array4 array aindex ELEMSIZE
         endcase
      endswitch
   endif
endproc


;***********************************************************************
;* This procedure checks for an array index out of range               *
;***********************************************************************

$ifdef ADEBUG
proc achkndx
intparm dim,val,max
   string errstr

        if((val < 0) || (val >= max))
      strfmt errstr "ERROR: Dimension %d array index out of range" dim
      message errstr
   endif
endproc
$endif


